import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:google_maps_flutter/google_maps_flutter.dart';

import '../../../gen/assets.gen.dart';
import '../../booking/widgets/place_result_item.dart';
import '../provider/place_lat_lng_fetcher_providers.dart';
import '../provider/search_place_providers.dart';
import '../provider/selected_loc_text_field_providers.dart';
import '../provider/way_point_list_providers.dart';
import 'place_confirm_sheet.dart';

class PlaceLookupStateView extends ConsumerWidget {
  final Widget? initialStateView;

  const PlaceLookupStateView({
    super.key,
    this.initialStateView,
  });

  @override
  Widget build(BuildContext context, WidgetRef ref) {
    final placeSearchState = ref.watch(searchPlaceNotifierProvider);
    final selectedLocationField = ref.watch(selectedLocTextFieldNotifierProvider);
    final wayPointNotifier = ref.read(wayPointListNotifierProvider.notifier);
    final wayPointsState = ref.watch(wayPointListNotifierProvider);
    return AnimatedSwitcher(
      duration: const Duration(milliseconds: 300),
      child: placeSearchState.when(
        data: (places) => NotificationListener<ScrollNotification>(
          onNotification: (scrollNotification) {
            FocusScope.of(context).unfocus();
            return false;
          },
          child: ListView.separated(
            padding: EdgeInsets.zero,
            itemCount: places.length,
            separatorBuilder: (context, index) => const Divider(
              thickness: 0.3,
              indent: 48,
              height: 16,
            ),
            itemBuilder: (context, index) {
              final place = places[index];
              return PlaceResultItem(
                onPressed: () async {
                  final LatLng? location =
                      await ref.read(placeLatLngFetcherNotifierProvider.notifier).getLatLngFromPlaceId(place.placeId);
                  if(context.mounted){
                    wayPointNotifier.updateWayPoint(
                      index: selectedLocationField ?? 0,
                      name: name(context, wayPointsState.length, selectedLocationField ?? 0),
                      address: place.subtitle,
                      location: location ?? const LatLng(0, 0),
                    );
                    ref.read(searchPlaceNotifierProvider.notifier).reset();
                  }

                },
                title: place.title,
                subtitle: place.subtitle,
                trailing: place.distance,
              );
            },
          ),
        ),
        error: (error, stackTrace) => Text(error.toString()),
        loading: () => Assets.lottie.loading.lottie(),
      ),
    );
  }
}
